#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/DBStore.py,v $
# Version:     $RCSfile: DBStore.py,v $ $Revision: 1.2 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
DBStore.py

Using the FileDevice class as a host, DBStore devices provide the
StoreImpl interface for persistent data storage.

"""

#############################################################################

import copy, os, pickle, random, sys

from   GnutOS.Exceptions         import HM_DeviceParameterError, \
                                        HM_DeviceError
from   GnutOS.Primitive          import true, nil
from   GnutOS.Soup               import Soup
from   GnutOS.VBO                import VBO
from   GnutOS.Classes            import Frame

from   GnutOS.Store              import StoreImpl, StoreException
from   GnutOS.DeviceDriver       import DeviceDriver
from   GnutOS.Device             import Device

from   GnutOS                    import FileDevice #fixme: dependency?


try:
    import bsddb
except ImportError:
    print "\n\nCannot find Berkeley DB module.\n"""
    sys.exit(1)


#############################################################################
#  Device Identifiers

TYPE = "store"
NAME = "DBStore"

MAGIC = "bsddbstore98v1"


#############################################################################
#  Exceptions

class ParameterError(HM_DeviceParameterError):
    """Bad parameters for DBStore initialisation."""
    pass

class DBStoreException(StoreException):
    """DBStore related exceptions."""
    pass

class NoSuchFileDevice(DBStoreException):
    """No FileDevice mounted with specified name."""
    pass

class DBError(DBStoreException):
    """The DBStore had an internal database error."""
    pass

class HostOSError(DBStoreException):
    """General error from host OS."""
    pass

class BadFormatOnFileDevice(DBStoreException):
    """Attempted to open a FileDevice formatted with something other
    than a DBStore."""
    pass

class BadFileDevice(DBStoreException):
    """An attempt to format a FileDevice failed because of the FileDevice."""
    pass


#############################################################################

def install(mgr):
    """Load and register the driver."""

    mgr.RegisterDriver(DBStoreDriver)

    return


#############################################################################

class DBStoreDriver(DeviceDriver):
    """File device-based GnutOS store."""

    def __init__(self, mgr):
	"""Initialise the DBStore driver.

	*mgr*      -- reference to the HardwareManager
	Returns    -- new driver instance

	This class is a device driver, mapping a File device into a
	DBStore device which is then able to be used by the
	StorageManager to back a Store object."""

	self._mgr = mgr
	return


    def Type(self):
	"""Return the general type of devices managed by this driver."""
	return TYPE


    def Name(self):
	"""Return the specific type of devices managed by this driver."""
	return NAME


    def HostDriverNames(self):
	"""Return a list of names of devices that can host this StoreImpl."""
	return [FileDevice.NAME, "DummyFile"]


    def ProbeDevice(self, dev_id):
	"""Probe a FileDevice to determine whether it hosts a Store.

	*dev_id*    -- identifier for a FileDevice instance
	Returns     -- integer store format version number
	Exceptions  -- ???

	Test the specified FileDevice to determine whether it has been
	formatted with a DBStore.  Returns a integer store format
	version number.  If the result is less than 1, the device does
	not contain a store."""

	#-- get the FileDevice
	ref_dev = self._mgr.Device(dev_id)

	#-- is the file a valid store?
	try:
	    db = bsddb.hashopen(ref_dev.HostName(), "w")

	except bsddb.error:
	    return 0

	except:
	    raise HostOSError(ref_dev.HostName())

	#-- FileDevice is a db file, now check the definition
	if not db.has_key("_magic") or db["_magic"] != MAGIC:
	    db.close()
	    return 0

	if not db.has_key("_version"):
	    db.close()
	    return 0

	ver = db["_version"]
	db.close()

	return ver


    def NewDevice(self, t_param):
	"""Create a new DBStore instance.

	*t_param* -- parameters for file store (see below).
	Returns   -- reference to the new device instance

	DBStore instances need one parameter -- a FileDevice
	instance already loaded by the Gnuton. Additional parameters
	are currently ignored."""

	#-- check parameters
	if len(t_param) != 1:
	    raise HM_DeviceParameterError(t_param)

	#-- lookup device
	file_dev_id = t_param[0]
        file_dev_ref = self._mgr.Device(file_dev_id)

	#-- create a new DBStore instance
	dev = DBStore(self, file_dev_ref)

	return dev


#############################################################################

class DBStore(StoreImpl, Device):
    """File device-based GnutOS store."""


    def __init__(self, ref_drv, ref_file):
	"""Initialise a file-based Store object.

	*ref_drv*  -- reference to the DBStoreDriver
	*ref_file* -- reference to the FileDevice hosting the DBStore
	Returns    -- new store object

	File-based stores are the default type of store under GnutOS.
	You specify the maximum size of the file, and the path name,
	and this class uses the Berkeley DB database to manage the
	soup and VBO entries in the store.

	This class does NOT support compression of the store data."""

	self._driver = ref_drv
	self._file = ref_file

	#-- check file device
	if self._file.DriverName() not in self._driver.HostDriverNames():
	    raise IncompatibleHostForStore(ref_drv, ref_file)

	#-- initialise Store super-class
	StoreImpl.__init__(self)

	#-- implementation attributes
	self._def = {}                # store definition frame
	self._d_soup = {}             # internal soup table

	self._def["name"] = ""
	self._def["wp"] = 0           # is this store write protected?
	self._def["rom"] = 0          # is this a ROM store?
	self._def["size"] = 0         # maximum allowed byte size
	self._def["soups"] = []       # component soup names
	self._def["next_id"] = 0      # next entry id

	self._def["vbo"] = {}         # VBOs on this store

	return


    def Format(self):
	"""Format a FileDevice with a new DBStore.

	Returns    -- (nothing)

	Creates the File files required to implement the Store.  This
	is not done by the constructor, since this class also uses
	existing Store files when re-mounting a Store."""

	#-- is the file a valid store?
	try:
	    self._db = bsddb.hashopen(self._file.HostName(), "c")

	except bsddb.error, value:
	    if value[0] == 2:      #-- no such file or directory
		raise BadFileDevice(self._file.HostName(), value[0], value[1])

	    else:
		raise HostOSError(self._file.HostName(), value[0], value[1])

	except:
	    raise HostOSError()


	#-- write magic number
	self._db["_magic"] = MAGIC
	self._db["_version"] = "2"

	#-- write Store info to database
	self._def["size"] = self._file.MaxSize()
	self._db["store"] = pickle.dumps(self._def)

	return


    def Erase(self):
	"""Erase all contents of the store."""

	for key in self._db.keys():
	    del self._db[key]
	return


    def Open(self):
	"""Open an existing DBStore from the named files.

	Returns    -- nothing
	Exceptions -- ?

	Opens a previously created Store."""

	#-- get file name from FileDevice
	str_file = self._file.HostName()

	#-- open database
	try:
	    self._db = bsddb.hashopen(str_file, "w")

	except:
	    str_err = sys.exc_info()[1][1]
	    raise DBError(str_file, str_err)

	#-- load store definition
	d_def = pickle.loads(self._db["store"])
	self._def.update(d_def)

	#-- load soup defns
	#-- initialise index cache
	return


    def Close(self):
	""" """
	self._db.close()


    def SetName(self, str_name):
	"""Set the name of this store."""

	self._def["name"] = str_name
	return


    def GetName(self):
	"""Return the name of this store."""
	return self._def["name"]


    def GetDevice(self):
	"""Return the FileDevice hosting this store."""
	return self._file


    def SetROM(self, flag):
	pass

    def SetWriteProtect(self, flag):
	pass



    def Soup(self):
	"""Return the Soup class for this type of Store."""
	return DBSoup

    def Index(self):
	"""Return the Index class for this type of Store."""
	return DBIndex

    def Entry(self):
	"""Return the Entry class for this type of Store."""
	return DBEntry


    #-- public Store methods

    def CreateSoup(self, soupName):
	"""Create a new soup within this store."""

	#-- create File soup object
	soup = DBSoup(self, soupName)
	self._db["soup_%s" % soupName] = pickle.dumps(soup)

	#-- add soup to store list
	self._def["soups"].append(soupName)
	self._d_soup[soupName] = soup

	return soup


    def GetSoupNames(self):
	"""Return list containing names of soups on the store."""

	return self._def["soups"]


    def GetSoup(self, soupNameString):
	"""Return the named soup, or nil."""

	if self._d_soup.has_key(soupNameString):
	    return self._d_soup[soupNameString]

	else:
	    return None


    def HasSoup(self, soupName):
	"""Does this Store contain a Soup with this name?"""

	return self._d_soup.has_key(soupName)


    def IsReadOnly(self):
	#fixme: must check for read-only media also!
	if self._def["wp"] or self._def["rom"]:
	    return 1

	else:
	    return None


    def IsValid(self):
	"""Returns true if the store can be used."""

	#fixme: should this be in the visible class?
	return


    def TotalSize(self):
	"""Returns total size in bytes allowed for store on device."""

	#fixme: this limit is not currently enforced!
	return self._size


    def UsedSize(self):
	"""Returns the number of bytes used in the store."""

	#fixme: use stat() to get file length
	return self._used


#############################################################################

class DBSoup:
    """

    DBSoup instances store information about the soup itself: its
    name, signature, info frame, the list of entries and the list of
    indexes.

    The user-visible Soup class uses a DBSoup instance to store its
    data, but the user is never really aware of the underlying
    implementation.

    """
    def __init__(self, ref_store, soupName):
	
	self._def = {}        #-- definition info
	self._info = {}       #-- app-assigned info slots

	self._lst_entry = []  #-- entry uids
	self._lst_idx = []    #-- index uids

	#-- populate initial definition info
	self._def["name"] = soupName
	self._def["username"] = ""
	self._def["ownerapp"] = ""
	self._def["ownerappname"] = ""
	self._def["userdesc"] = ""
	self._def["signature"] = 0
	self._def["nextuid"] = 0
	self._def["indexmodtime"] = 0
	self._def["infomodtime"] = 0
	return


    def __getinitargs__(self):
	"""Internal method. """
	return (self._def["name"])


    def SetName(self, soupNameString):
	"""Set the name of the soup to the supplied string.

	*soupNameString* -- string name for soup
	Returns          -- new name

	This is really just a consistency checking thing: the useful
	name of the soup is its key in the store database.  However,
	if the store key and this name are different, then at least
	you know you're in trouble.  """

	self._def["name"] = soupNameString
	return soupNameString


    def SetSignature(self, signature):
	"""Set the soup signature to the specified integer.

	*signature* -- integer signature for soup
	Returns     -- new signature

	Soup signatures are random integers assigned when the soup is
	created, used to differentiate between same-named soups.  """

	self._def["sig"] = signature
	return signature


    def GetName(self):
	"""Return the name of the soup."""
	return self._def["name"]


    def GetSize(self):
	"""Return the size of the soup in bytes.

	This is actually non-trivial: a soup consists of the soup
	itself, all indexes, and all entries.  Actually adding this
	info up would be expensive.  Why bother (yet) ? """

	#fixme: should do this ...
	return 0


    def GetNextUid(self):
	return self._def["nextuid"]


    def GetSignature(self):
	return self._def["signature"]


    #-- entry methods

    def AddEntry(self, uid):
	""" """
	self._lst_entry.append(uid)
	#fixme: where, exactly, should i enforce the index update?
	return


    def RemoveEntry(self, uid):
	""" """
	self._lst_entry.remove(uid)
	#fixme: where, exactly, should i enforce the index update?
	return


    def RemoveAllEntries(self):
	"""Remove all entries from soup."""

	for uid in self._lst_entry:
	    self.RemoveEntry(uid)
	return


    #-- index methods

    def AddIndex(self, uid):
	""" """
	self._lst_idx.append(uid)
	return

    def RemoveIndex(self, uid):
	""" """
	self._lst_idx.remove(uid)
	return


    def GetIndexes(self):
	return copy.copy(self._lst_idx)


    def IndexSizes(self):
	#fixme: what does this do?
	return 0


    def GetIndexesModTime(self):
	return self._def["indexmodtime"]


    #-- tags methods

    def AddTags(self, tags):
	return

    def ModifyTag(self, oldTag, newTag):
	return

    def RemoveTags(self, tags):
	return

    def GetTags(self):
	return

    def HasTags(self):
	return

    #-- info methods

    def SetAllInfo(self, frame):
	"""Set the soup's info slot to the supplied frame."""

	self._info = copy.copy(frame)    #fixme: copy? deepcopy?
	return


    def SetInfo(self, slotSymbol, value):
	"""Set the value of a slot in the soup's info frame."""

	self._info[slotSymbol] = value
	return nil


    def GetAllInfo(self):
	"""Returns all contents of the soup info frame."""
	return copy.copy(self._info)


    def GetInfo(self, slotSymbol):
	"""Return the contents of the specified slot in the info frame.

	*slotSymbol* -- string, slot name in info frame
	Returns      -- nil, or contents of named slot

	If the specified slot does not exist, this method returns
	*nil*."""

	if self._info.has_key(slotSymbol):
	    return self._info[slotSymbol]

	else:
	    return nil


    def GetInfoModTime(self):
	return self._def["infomodtime"]


    def MakeKey(self, string, indexPath):
	""" """
	return

    def Flush(self):
	""" """
	return



#############################################################################

class DBIndex:
    """ """
    def __init__(self, store):
	""" """
	self._d_entry = {}
	return


class DBEntry:
    """ """
    def __init__(self, frame):
	""" """
	self._d_frame = frame
	return


#############################################################################
