#! /usr/bin/env python
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/DeskPad.py,v $
# Version:     $RCSfile: DeskPad.py,v $ $Revision: 1.3 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
DeskPad

The DeskPad device -- a GNUton for desktop systems.

This file implements the device-specific components of a desktop
GNUton, wrapping the Gnuton instance itself, and providing drivers for
the screen, mouse, keyboard, etc.

"""

#############################################################################

import string, sys
from   Tkinter             import *
from   GnutOS.Gnuton       import Gnuton
from   GnutOS.Classes      import Binary, Symbol


#############################################################################

BOOTMSG = """DeskPad v%s (%s)
Copyright (C) 1998-%s, David Arnold.

Loading GnutOS ... """

REVISION = "$Revision: 1.3 $"[11:-2]
VERSION  =  "0.1"
YEAR     =  "$Date: 2000/09/25 11:20:32 $"[7:11]

usage    = """Usage: %s config_file\n\n"""

#############################################################################

class DeskPad:
    """

    A GNUton device has two fundamental components: a Gnuton instance
    which implements the GnutOS, and provides the generic nature of
    the machine; and a wrapper which maps the specific hardware onto
    that generic kernel.

    This class is the wrapper.

    It reads a configuration file to determine the properties of the
    emulated device, and creates a boot block describing them in the
    form required by the generic Gnuton class.

    The DeskPad is the first (and maybe only) hardware to support
    Gnuton.  It is intended to run on desktop personal computers, with
    some hardware expectations built in as a result of that.

    """

    def __init__(self, d_conf):
	"""Create a DeskPad instance.

	*d_conf*   -- dictionary: the boot block
	Returns    -- only once it is finished ;-)
	Exceptions -- any exceptions raised at this level represent a
	              catastrophic failure of the emulator.

        The DeskPad provides a set of devices used by the Gnuton:
        power, reset, backlight, screen, digitiser, keyboard, file
        stores, memory, etc.  Each of these are implemented as a
        device class, and passed to the Gnuton via the boot block.

	Also implemented are the Debug and Status devices."""

	#-- save configuration
	self._conf = d_conf

	#-- create debugging devices
	self._debug = Inspector(self)
	self._stats = Status(self)

	self.log(BOOTMSG % (VERSION, REVISION, YEAR))
	self.log("configuration from %s\n" % d_conf["config"])

	#-- initialise basic devices (and update config)
	#fixme: the Gnuton can read specific device classes from config
	#fixme: but needs to have info about Tk root, etc

	self.tk = Tk()
	self._conf["DPDisplay"][0].append(self.tk)

	self._backlight = 0           #-- is the backlight on?
	self._orientation = 0         #-- 0 vertical, 1 horizontal

	#-- create GUI
	self.gui()

	#-- create GNUton instance (with DeskPad as device instance)
	self._gnut = Gnuton(self)

	#-- load "ROM" applications
	#-- load "RAM" applications

	#-- run
	self._gnut.run()

	#-- clean up devices
	self._gnut.destroy()
	self.log("DeskPad exiting.\n")

	#-- exit
	return


    def log(self, txt, level=10):
	self._debug.log(txt, level)
	return


    def config(self):
	return self._conf


    def gui(self):
	"""Create the DeskPad user interface."""

	#top_status = Toplevel(self.tk)
	#top_inspect = Toplevel(self.tk)
	#top_main = Toplevel(self.tk)

	#-- create main menu bar
	frm_menu = Frame(self.tk)

	mb_file = Menubutton(frm_menu, text="File", underline=0)
	mb_edit = Menubutton(frm_menu, text="Edit", underline=0)
	mb_gnut = Menubutton(frm_menu, text="Special", underline=0)
	mb_help = Menubutton(frm_menu, text="Help", underline=0)

	menu_file = Menu(mb_file, tearoff=0)
	menu_file.add(COMMAND, label="New",  underline=0)
	menu_file.add(SEPARATOR)
	menu_file.add(COMMAND, label="Open ...", underline=0)
	menu_file.add(COMMAND, label="Save", underline=0)
	menu_file.add(COMMAND, label="Save as ...", underline=5)
	menu_file.add(SEPARATOR)
	menu_file.add(COMMAND, label="Print ...", underline=0)
	menu_file.add(SEPARATOR)
	menu_file.add(COMMAND, label="Exit", 
		      underline=1, 
		      command=self.exit)
	mb_file.config(menu=menu_file)

	menu_edit = Menu(mb_edit, tearoff=0)
	menu_edit.add(COMMAND, label="Undo",  underline=0)
	menu_edit.add(SEPARATOR)
	menu_edit.add(COMMAND, label="Cut",  underline=0)
	menu_edit.add(COMMAND, label="Copy",  underline=0)
	menu_edit.add(COMMAND, label="Paste",  underline=0)
	menu_edit.add(SEPARATOR)
	menu_edit.add(COMMAND, label="Preferences ...",  underline=0)
	mb_edit.config(menu=menu_edit)

	menu_gnut = Menu(mb_gnut, tearoff=0)
	menu_gnut.add(COMMAND, label="Mount Store ...",  underline=0)
	menu_gnut.add(COMMAND, label="Unmount Store ...",  underline=0)
	menu_gnut.add(COMMAND, 
		      label="Load package ...",  
		      command=self.load_package_dialog,
		      underline=0)
	menu_gnut.add(COMMAND, 
		      label="Toggle backlight",  
		      underline=0,
		      command=self.toggle_backlight)
	menu_gnut.add(SEPARATOR)
	menu_gnut.add(COMMAND, label="Reset ...",  underline=0)
	mb_gnut.config(menu=menu_gnut)

	menu_help = Menu(mb_help, tearoff=0)
	menu_help.add(COMMAND, label="About ...",  underline=0)
	mb_help.config(menu=menu_help)

	mb_file.pack(side=LEFT)
	mb_edit.pack(side=LEFT)
	mb_gnut.pack(side=LEFT)
	mb_help.pack(side=RIGHT)
	frm_menu.pack(fill=X, expand=1)

	#-- screen area
	frm_screen = Frame(self.tk)
	self._screen = Canvas(frm_screen, width=480, height=640)
	self._screen.pack()
	frm_screen.pack()

	#-- status bar
	frm_status = Frame(self.tk)
	frm_status.pack()

	self.tk.update()



    def toggle_backlight(self):
	"""Toggle the screen backlight."""

	if self._backlight:
	    self._backlight = 0
	    self._screen.config(bg="gray70")

	else:
	    self._backlight = 1
	    self._screen.config(bg="#000088880000")

	return


    def exit(self):
	"""Stop the event loop of the Gnuton."""

	self._gnut._run = 0  #fixme: need API to do this ...
	return


    def load_package_dialog(self):
	"""Dialog allowing selection of a package file to load."""

	self._pkg_dialog = PackageLoader(self.tk, self.load_package_from_file)
	return


    def load_package_from_file(self, file):
	"""Load a package from the specified file."""

	self._gnut.log("Trying package %s: " % file)

	#-- read data
	try:
	    f = open(file)
	    s = f.read()
	    f.close()
	    self._gnut.log("Found.\n")

	except:
	    self._gnut.log("No such file!\n")
	    self._pkg_dialog.close()
	    return

	#-- get soup and default store
	def_store = self._gnut._sm.GetDefaultStore()
	pkg_soup = self._gnut._sm.GetUnionSoupAlways("packages")

	#-- create binary object
	vbo = def_store.NewVBO(Symbol("package"), len(s))
	BinaryMunger(b, 0, nil, s, 0, nil)

	#-- create a new entry in the packages soup
	frm = {Symbol("pkgRef"): b, Symbol("packageName"): nil}
	entry = pkg_soup.AddToDefaultStoreXmit(Frame(frm), nil)
	entry[Symbol("pkgRef")] = b

	#-- write VBO to store
	try:
	    EntryChangeXmit(entry, nil)

	except evt.ex.fr.store:
	    Notify(kNotifyAlert, "Package Loader", "Problem saving package.")
	    return

	#-- call package creation method
	paramFrame = {"callback": self._pkg_dialog.callback,
		      "callbackFrequency": 500}
	pkg_ref = ref_store.SuckPackageFromBinary(b, paramFrame)

	#-- create templates
	#-- install app symbol in root frame
	#-- add to extras drawer
	#-- run install script

	#fixme: this is just debugging -- kill it
	import time
	self._pkg_dialog.setPackage("Gnuton Package", 5, None)
	for i in range(5):
	    self._pkg_dialog.setPart(i+1)
	    self.tk.update()
	    time.sleep(2.0)
	    self._pkg_dialog.progress((i+1)*20)

	#-- close dialog
	self._pkg_dialog.close()

	return


#############################################################################

class PackageLoader:
    def __init__(self, tkroot, callback):
	"""Load a package from a file."""

	self._top = Toplevel(tkroot)
	self._cb = callback

	#-- string variables
	self._file = StringVar(tkroot)
	self._file.set("<file>")
	self._pkg_name = StringVar(tkroot)
	self._pkg_name.set("<package>")
	self._parts = StringVar(tkroot)
	self._parts.set("Part n of m")

	frm_file = Frame(self._top)
	frm_frozen = Frame(self._top)
	frm_pkg = Frame(self._top)
	frm_text = Frame(frm_pkg)
	frm_prog = Frame(self._top, relief=FLAT)
	frm_but = Frame(self._top)

	lb_file = Label(frm_file, text="File")
	txt_file = Entry(frm_file, textvariable=self._file)
	but_file = Button(frm_file,
			  text="Browse ...",
			  #command=self._browse_file,
			  state="disabled")   #fixme:  enable file browsing

	cb_frozen = Checkbutton(frm_frozen, 
				text="Do not activate package after installation (leave frozen).",
				highlightthickness=0)

	lb_pkg = Label(frm_text, textvariable=self._pkg_name)
	lb_part = Label(frm_text, textvariable=self._parts)
	lb_icon = Label(frm_pkg)

	can_prog = Canvas(frm_prog, height=15)

	but_install = Button(frm_but, 
			     text="Install",
			     command=self.install)
	but_cancel = Button(frm_but, 
			    text="Cancel",
			    command=self.close)

	#-- pack widgets
	frm_file.pack(fill=X, expand=1)
	lb_file.pack(side=LEFT)
	txt_file.pack(side=LEFT, fill=X, expand=1, padx=5)
	but_file.pack(side=LEFT)

	frm_frozen.pack(fill=X, expand=1)
	cb_frozen.pack(side=LEFT)

	frm_pkg.pack()
	frm_text.pack()
	lb_pkg.pack(side=TOP, fill=X)
	lb_part.pack(side=TOP, fill=X)
	lb_icon.pack(side=TOP, fill=BOTH)

	frm_prog.pack(side=TOP, fill=X, expand=1)
	can_prog.pack(side=TOP, fill=X, expand=1, padx=10, pady=6)

	frm_but.pack(fill=X, expand=1)
	but_install.pack(side=LEFT, expand=1)
	but_cancel.pack(side=LEFT, expand=1)

	#-- set title
	Wm.title(self._top, "Load Package")

	#-- setup progress bar
	self._prog_can = can_prog
	self._prog_bar = can_prog.create_rectangle(0, 0, 1, 15,
						   fill="blue",
						   width=0)
	return


    def close(self):
	"""Remove the dialog."""
	Wm.withdraw(self._top)
	return


    def install(self):
	"""Install the selected package file."""
	apply(self._cb, (self._file.get(),))
	return


    def progress(self, percent=0):
	"""Set the progress bar to a percentage complete."""

	w = self._prog_can.configure("width")[4]
	l = int(int(w) * percent / 100.0)
	self._prog_can.coords(self._prog_bar, 0, 0, l, 15)
	return


    def setPackage(self, name, nparts, icon):
	"""Set package info."""

	self._pkg_name.set(name)

	self._nparts = nparts
	self._curpart = 0

	self._parts.set("Part %d of %d" % (self._curpart, self._nparts))
	return


    def setPart(self, part):
	self._curpart = part
	self._parts.set("Part %d of %d" % (self._curpart, self._nparts))
	return


    def callback(self, callbackInfo):
	"""Callback method for SuckPackageFromBinary."""

	print "callback"
	return



class FileChooser:
    def __init__(self, tkroot, callback):

	self._top = Toplevel(tkroot)
	self._cb = callback

	b = Button(self._top, text="Load Me", command = self.result)
	b.pack()
	return

    def result(self):
	self._cb("/home/arnold/play/gnuton/test/newt-test.pkg")


#############################################################################

class Status:
    def __init__(self, ref_dp):
	pass


#############################################################################\

class Inspector:

    def __init__(self, ref_dp):
	self._level = 0
	self._console = sys.stderr


    def log(self, txt, level):
	"""Write a message to the system console.

	*txt*      --   string, text to write to console
	*level*    --   integer, used for filtering. 0 unimportant, 10 very
	Returns    -- (none)
	Exceptions -- (none)"""

	if not self._console:
	    return

	if level >= self._level:
	    self._console.write("%s" % txt)

	return



#############################################################################

if __name__ == '__main__':
    d_conf = {}

    #-- parse command args
    if len(sys.argv) == 2:
        str_config = sys.argv[1]
    else:
        str_config = "test.gnt"
        
    #-- load config file
    if not str_config:
        sys.stderr.write(usage % sys.argv[0])
        sys.exit(1)

    d_conf["config"] = str_config

    try:
        f_conf = open(str_config)

    except:
        sys.stderr.write(usage % sys.argv[0])
        sys.exit(1)

    lst_conf = f_conf.readlines()
    f_conf.close()

    #-- parse config
    cnt = 0
    not_ok = 0

    for line in lst_conf:
	cnt = cnt + 1
	line = string.strip(line)
	if line[0:1] == "#" or line == "":
	    pass

	else:
	    lst_tok = string.split(line)
	    if len(lst_tok) < 2:
		print "Error: Line %d -- no parameters for command." % cnt
		not_ok = 1

	    elif lst_tok[0] == "driver":
		drv_type = lst_tok[1]
		if not d_conf.has_key(None):
		    d_conf[None] = [drv_type]
		else:
		    d_conf[None].append(drv_type)

	    elif lst_tok[0] == "device":
		dev_type = lst_tok[1]
		if not d_conf.has_key(dev_type):
		    d_conf[dev_type] = [lst_tok[2:]]
		else:
		    d_conf[dev_type].append(lst_tok[2:])

	    else:
		print "Error: Line %d -- unknown command %s."%(cnt,lst_tok[0])
		not_ok = 1

    if not_ok:
	sys.exit(1)

    #-- override file using command line args

    #-- create DeskPad instance
    dp = DeskPad(d_conf)

    #-- exit nicely
    sys.exit(0)


#############################################################################


