#! /usr/bin/env python
#############################################################################
#
# Project:     NewtonScript Interpreter
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/nvm.py,v $
# Version:     $RCSfile: nvm.py,v $ $Revision: 1.1.1.1 $
# Copyright:   (C) 1997-1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""

implementation of the NewtonScript bytecodes, using the Python
datatype classes and stack.

"""

#############################################################################

from    vm   import VM



#############################################################################

class NewtonVM(VM):
    """NewtonScript virtual machine operations.

    For all bytecodes, the method is called with two arguments: a
    reference to the interpreter "self", and an optional argument
    decoded from the instruction named "arg".  The use of "arg" is
    dependent on the bytecode.

    "arg" is passed in raw form -- it is not converted into a Python
    datatype prior to invocation of the method."""


    def _BC_POP(self, arg):
	"""Removes the top item from the stack.

	"arg"           (not used)."""

	self._stack.pop()
	return


    def _BC_DUP(self, arg):
	"""Duplicates the top item of the stack.

	"arg"           (not used)."""

	self._stack.push(self._stack.top())
	return


    def _BC_PUSHIMMEDIATE(self, arg):
	"""Push immediate value on stack.

	"arg"           a NewtonScript immediate value

	This instruction is used to push any immediate value that can
	be represented in 16 (sign extended) bits or less.  Values
	that require the full 32 bits are stored as literals and
	pushed using BC_PUSHLITERAL.

	Some examples of the use of this instruction:

	20		push 0
	22		push Nil
	23		push @0
	24		push 1
	27 00 1A	push True
	27 02 86	push $(
	27 FF F8	push -2

	"""

	self._stack.push(ns2py(arg))
	return


    def _BC_PUSHLITERAL(self, arg):
	"""Push "arg"th element of literals array on stack.

	"arg"           (integer) index into literals array
	Return value    (none)

	"""

	self._stack.push(self._frame.literals[ns2py(arg)])
	return


    def _BC_PUSHLOCAL(self, arg):
	"""Push "arg"th element of arguments frame on stack.

	This bytecode is commonly used to push either the value of an
	argument to the function or that of a local variable.

	Example:

	func( foo ) return foo translates to:

	73	push value of foo (remember argument slots start
				   after the three fixed slots in
				   argFrame, hence the index is 3 not
				   0).
	02	Return

	"""

	key = self._frame.argFrame_idx[arg]
	self._stack.push(self._frame.argFrame[key])
	return


    def _BC_PUSHSLOTVALUE(self, arg):
	"""

	This takes a frame and the name of a slot in that frame and
	pushes the value contained in the slot.  If the slot name is a
	single name then it is represented by a symbol reference.  If
	the slot name is a path expression (e.g. foo.bar) then it is
	represented by an array of class pathExpr with one element per
	section of the path.  Each element is then a reference to the
	symbol for that part of the path.

	Example:

	return self.foo.bar translates to:

	03	push self
	18	push '[pathExpr: 'foo, 'bar] (assuming it to be
					      the first literal)
	91	Get slot value
	02	Return

	"""

	pass


    def _BC_PUSHEXTERNAL(self, arg):
	"""

	This bytecode pushes onto the stack the value contained in a
	variable or slot from outside of the function.  The name of
	the variable to push is given by literal "arg" in the
	function's literals array; this literal should be a symbol
	reference.

	Example:

	length( functions ) translates to:

	70	push value of functions (assuming 'functions is the
					 first entry in	the literals
					 array of the containing CodeBlock)
	C7 00 12	Return the length of functions.

	"""

	sym = 0
	pass


    def _BC_ASSIGNTOSLOT(self, arg):
	"""

	This assigns the value Y to the slot SlotName of frame X,
	i.e. X.SlotName := Y.  As with 221 (push slot value) the slot
	name is represented either by a simple symbol reference or a
	pathExpr array for a complex path expression.

	"""

	pass


    def _BC_ASSIGNTOSLOTANDPUSH(self, arg):
	"""

	Identical to 230 except that the value assigned to the slot is
	also pushed onto the stack.

	"""

	pass


    def _BC_ASSIGNTOLOCAL(self, arg):
	"""

	Value X is assigned to the "arg"th slot of the CodeBlock's
	argFrame frame.  In other words, this bytecode is usually used
	to set the value of a local variable.

	Example:

	local x := '[] translates to:

	18	push '[] (assuming it to be the first literal)
	A7 00 10	Assign to x (assuming it to be 16th slot in argFrame)

	"""

	pass


    def _BC_ASSIGNTOEXTERNAL(self, arg):
	"""

	Value X is assigned to a variable or slot from outside of the
	function.  The name of the variable to assign to is given by
	literal N in the function's literals array; this literal
	should be a symbol reference.

	"""

	pass


    def _BC_SELF(self, arg):
	"""Push reference to current frame.

	Pushes a reference to the frame containing the current
	function onto the stack.  In other words this pushes the value
	represented by the NewtonScript value self."""

	pass


    def _BC_ADD(self, arg):
	"""Addition of integers and floats.

	x = pop(); y = pop(); push (x+y)

	The numbers can be integers or floats.

	The 30N bytecode series encodes a number of different
	low-level system function calls.  Any NewtonScript operator or
	built-in function not explicitly mentioned here is implemented
	by using a standard function call (bytecode 05N).  This first
	bytecode of the series implements addition (of integers or
	floats).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x+y)

	return


    def _BC_SUB(self, arg):
	"""Subtraction of integers or floats.

	x = pop(); y = pop(); push (x-y)
	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x-y)

	return


    def _BC_MULT(self, arg):
	"""Multiply two numbers.

	x = pop(); y = pop(); push (x*y)
	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x*y)

	return


    def _BC_FDIV(self, arg):
	"""Divide two numbers with float result.

	x = pop(); y = pop(); push (x/y)

	This implements division (of integers or floats) with a float
	result, i.e.  it corresponds to the <#007F>/ operator in
	NewtonScript.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(float(x)/y)

	return


    def _BC_IDIV(self, arg):
	"""

	This implements division of integers with an integer result,
	i.e. it corresponds to the div operator in
	NewtonScript.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(int(float(x)/y))

	return


    def _BC_NOT(self, arg):
	"""

	This takes any value X and returns not X.  Not Nil is True,
	not anything else is Nil.

	"""

	#-- pop arguments
	x = self._stack.pop()

	#-- test
	if x == NS_NIL:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_COMPEQ(self, arg):
	"""


	This compares X and Y for equality and pushes a Boolean result (True
	or Nil).

	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x == y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)

	return


    def _BC_COMPNEQ(self, arg):
	"""

	This compares X and Y for inequality and pushes a Boolean
	result (True or Nil).

	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x != y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_COMPLT(self, arg):
	"""

	This checks if X is less than Y and pushes a Boolean result
	(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x < y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_COMPGT(self, arg):
	"""

	This checks if X is greater than Y and pushes a Boolean result
	(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x > y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_COMPLE(self, arg):
	"""

	This checks if X is less than or equal to Y and pushes a
	Boolean result(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x <= y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_COMPGE(self, arg):
	"""

	This checks if X is greater than or equal to Y and pushes a
	Boolean result (True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x >= y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def _BC_BAND(self, arg):
	"""

	This performs a binary and of the two integers X and Y.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	self._stack.push(x & y)


    def _BC_BOR(self, arg):
	"""

	This performs a binary or of the two integers X and Y.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	self._stack.push(x | y)


    def _BC_BNOT(self, arg):
	"""

	This performs a binary not of the integer X.

	"""
	#-- pop arguments
	x = self._stack.pop()
	self._stack.push(~x)


    def _BC_CALLGLOBAL(self, arg):
	"""Call global function.

	"arg" arguments to the function are expected on the stack
	(pushed in the same order that they are listed in the function
	parameter list), followed by a reference to the symbol that
	represents the name of the function (this is stored in the
	literals array and pushed using 03N).  Although the stack
	description indicates that the instruction itself pushes
	nothing onto the stack, remember that all NewtonScript
	functions conventionally push a single item onto the stack
	before returning.

	Example:

	max( foo, bar ) translates to:

	7B	push foo  (assuming it to be the first parameter of the
			   function) 
	7C	push bar  (assuming it to be the second)
	18	push 'max (assuming it to be the first literal in the
			   function)
	2A	Call a global function with two arguments.

	"""

	pass


    def _BC_CALLWITH(self, arg):
	"""Call function with 

	This directly translates the NewtonScript <#007F>call ... with
	() construct.  "arg" arguments to the function are expected
	on the stack (pushed in the same order that they are listed in
	the function parameter list), followed by a reference to a
	CodeBlock frame that has had the BC_SELF instruction called on
	it.  As before, remember that all NewtonScript functions
	conventionally push a single item onto the stack before
	returning.

	Example:

	call func(foo) return foo with (1) translates to:

	24	push 1
	18	push func (assuming it to be the first literal in the
			   containing function)
		Note that inline function definitions like this are
		encoded as a CodeBlock frame in the literals array of
		the containing function.
	04	Form a closure for the func (?)
	31	Call ... with passing one argument.

	"""

	pass


    def _BC_RETURN(self, arg):
	"""Exits from a function.

	By convention, the top item on the stack is the return value
	of the routine.  It is possible to return from a routine with
	more items on the stack (or none at all), however the current
	Apple NewtonScript compiler ensures that the functions that it
	produces always leave one and only one value on the stack."""

	pass


    def _BC_CLOSURE(self, arg):
	"""Create closure.

	This bytecode is inserted by the compiler whenever it has just
	pushed a CodeBlock frame either as a parameter to a function
	call or message send, or as the target of a call...with
	statement.  My guess is that this initialises the three
	reserved fields in the argFrame of the CodeBlock (see 1.1),
	but I haven't yet attempted to verify this."""

	pass


    def _BC_SENDMESSAGE(self, arg):
	"""Send Message.


	This directly translates the NewtonScript <#007F>: construct.  N
	arguments to the message are expected on the stack (pushed in the same
	order that they are listed in the function parameter list), followed
	by a reference to the receiving frame and then the symbol of the
	message.  Again, remember that all NewtonScript functions
	conventionally push a single item onto the stack before returning.

	Example:

	:Open() translates to:

	03	push self
	18	push 'Open (assuming it to be the first literal in
			    the function)
	38	Send message with zero parameters.

	"""

	pass


    def _BC_CONDSENDMESSAGE(self, arg):
	"""

	This directly translates the NewtonScript <#007F>:? construct.
	"arg" arguments to the message are expected on the stack
	(pushed in the same order that they are listed in the function
	parameter list), followed by a reference to the receiving
	frame and then the symbol of the message to be sent if the
	given function exists.  If the function doesn't exist, Nil is
	placed on the stack else it's left up to the function to put
	something there.

	Example:

	child:?DoItIfYouCan() translates to:

	7B	push child (assuming it to be the first parameter of
			    the function)
	18	push 'DoItIfYouCan (assuming it to be the first literal
				    in the function)
	40	Send message with zero parameters if possible.

	"""

	pass


    def _BC_INHSENDMESSAGE(self, arg):
	"""

	This directly translates the NewtonScript <#007F>inherited:
	construct.  "arg" arguments to the message are expected on the
	stack (pushed in the same order that they are listed in the
	function parameter list), followed by the symbol of the
	message.  Again, remember that all NewtonScript functions
	conventionally push a single item onto the stack before
	returning.

	Example:

	inherited:Open() translates to:

	18	push 'Open (assuming it to be the first literal in the
			    function)
	48	Send message with zero parameters to inheritance parent.

	"""

	pass


    def _BC_CONDINHSENDMESSAGE(self, arg):
	"""

	This directly translates the NewtonScript <#007F>inherited:?
	construct.  "arg" arguments to the message are expected on the
	stack (pushed in the same order that they are listed in the
	function parameter list), followed by the symbol of the
	message to be sent if the given function exists.  If the
	function doesn't exist, Nil is placed on the stack else it's
	left up to the function to put something there.

	Example:

	inherited:?DoItIfYouCan() translates to:

	18	push 'DoItIfYouCan (assuming it to be the first literal
				    in the function)
	50	Send message with zero parameters to inheritance parent
		(if possible).

	"""

	pass



    def _BC_GOTO(self, arg):
	"""

	This instruction causes execution to pass to location "arg",
	where "arg" is the offset from the start of the block of
	bytecode (i.e. gotos are absolute, not relative).

	"""

	pass


    def _BC_GOTOIFNIL(self, arg):
	"""

	X is removed from the stack and examined.  If it is not Nil,
	execution continues with the next instruction in sequence.  If
	it is Nil, execution passes to location "arg" where "arg" is
	the offset from the start of the block of bytecode.

	"""

	pass


    def _BC_GOTOIFNOTNIL(self, arg):
	"""

	X is removed from the stack and examined.  If it is Nil,
	execution continues with the next instruction in sequence.  If
	it is not Nil, execution passes to location "arg" where "arg"
	is the offset from the start of the block of bytecode.

	"""

	pass


    def _BC_CREATEFRAME(self, arg):
	"""

	This bytecode creates a frame given "arg" slot values on the
	stack plus a frame map describing the slots in the frame.  The
	frame map is an array whose first item is Nil and then each
	subsequent item is a reference to a symbol giving the name of
	the corresponding slot in the frame being created.  The values
	are pushed onto the stack in the order of the slot names in
	the frame map.  If this is as clear as mud, maybe the example
	will help:

	Example:

	local fr := { size: 4, count: 7 } translates to:

	27 00 10	push 4
	27 00 1C	push 7
	18	push frame map (assuming it to be the first literal
				in the CodeBlock)
		The frame map is an array thus: [Nil, 'size, 'count]
	82	Create a frame with two slots
	A6	Store in fr (assumed to be the 6th slot in argFrame)

	A final NB.  The frame map array isn't a normal array of class
	Array.  It's class is a small integer, whose value seems to be
	made up from three possible flags.  I'm investigating the
	meaning of these flags at the moment.

	"""

	pass


    def _BC_CREATEARRAY(self, arg):
	"""

	This bytecode creates an array given N element values on the
	stack plus a reference to a symbol defining the class of the
	array.  The class of the array is specified in NewtonScript
	with an initial <#007F><class>: inside the array, e.g.:
	[stepChildren:].  If no class is specified, then the default
	class of Array is used.

	In the special case of N being 0xFFFF, the instruction takes a single
	NewtonScript integer from the stack instead of N element values and
	creates an empty array (i.e. each element is Nil) with the given
	number of elements.

	Example:

	local foo := [ 4, 7 ] translates to:

	27 00 10	push 4
	27 00 1C	push 7
	18	push 'Array (assuming it to be the first literal
			     in the CodeBlock)
	8A	Create an array with two elements
	A5	Store in foo (assumed to be the 5th slot in argFrame)

	"""

	pass


    def _BC_ADDARRAYSLOT(self, arg):
	"""

	This implements the AddArraySlot function e.g. AddArraySlot(
	foo, 2 ). Y is added as the last element of the array X.

	"""
	pass


    def _BC_MAKESTRING(self, arg):
	"""

	This takes an array of strings and combines them into a single
	string.  This function is used to implement the NewtonScript
	operators <#007F>& and <#007F>&&.  <#007F>&& is currently done
	by inserting an extra single space string into the array of
	strings to be translated.

	Example:

	fred && "me" translates to:

	70		push value of fred ('fred is the first literal)
	19 1A 1B	push " " (second), "me" (third) and 'Array (fourth)
	8B		Make an array from 3 parameters and a class symbol
	C7 00 16	Make a string from the array

	"""
	pass


    def _BC_DEREF(self, arg):
	"""

	This pushes the array element Y of array X.  Y should be an integer.

	"""

	pass


    def _BC_ASSIGNTOARRAYANDPUSH(self, arg):
	"""

	This assignes value Z to the pushes the array element Y of
	array X i.e.  X[Y] := Z.  Y should be an integer.

	"""

	pass


    def _BC_LENGTH(self, arg):
	"""

	This implements the length function.

	"""
	pass


    def _BC_CLONE(self, arg):
	"""

	This implements the clone function.

	"""
	pass


    def _BC_SETCLASS(self, arg):
	"""

	This implements the SetClass function e.g. SetClass( data,
	'Binary ).  X is a complex data structure (frame, array or
	data block); SetClass sets the class of that structure and
	then returns a reference to the now modified structure.

	"""
	pass


    def _BC_CLASSOF(self, arg):
	"""Implements the NewtonScript ClassOf function.

	x = pop(); push (ClassOf(x))
	"""

	x = self._stack.pop()
	self._stack.push(x.ClassOf())


    def _BC_SLOTEXISTS(self, arg):
	"""

	This implements the exists operator, but only when checking
	for the existance of a slot in a frame (e.g. foo.bar exists).
	When checking for the existance of a variable, a function call
	(bytecode 05N) to the function HasVar is used instead.

	"""
	pass


    def _BC_ONEXCEPTION(self, arg):
	"""

	This takes N pairs of an exception symbol and an offset into the
	bytecode to jump to if that exception occurs.  This is used to encode
	the high level try ... onexception construct.  Note that unlike all
	the goto bytecodes, the offset is encoded as a standard NewtonScript
	integer.  At the end of both the normal and the exception code, the
	bytecode 007 000 007 is executed - I guess it clears whatever state
	was set up.

	Example:

	try 1/0 onexception |evt.ex.div0| do nil becomes:

	18		push '|evt.ex.div0| (first literal)
	27 00 40	push integer 16
	C9		onexception '|evt.ex.div0| goto 16
	24 20		push 1, push 0
	C7 00 08	1/0
	07 00 07	End exception block (?)
	5F 00 14	Goto 20 (i.e. the end of this block)
	16: 22		Push Nil
	07 00 07	End exception block (?)

	"""
	pass


    def _BC_ENDEXCEPTION(self, arg):
	"""End exception.

	See _bc_onexception() for full explanation."""

	pass


    def _BC_FOREACH(self, arg):
	"""

	This is the fundamental instruction used to implement all
	NewtonScript foreach loops.  If the Boolean is Nil it begins a
	normal foreach loop and if it is True it begins a foreach
	deeply loop.  The iterator returned is then passed to the
	other two related instructions 005 (foreach next) and 006
	(foreach complete).  The iterator itself is an array
	containing various useful state information about what is
	being iterated over and how far the iteration has progressed.

	005 (foreach next) takes the iterator and modifies it so that
	the next item in the iteration becomes the current item.  006
	(foreach complete) checks the state of the iterator and
	returns True if all items have been iterated over and Nil if
	not.

	So, a foreach loop is implemented in bytecode as:

	??		push thing being iterated over
	27<#0102>00<#0102>1A/22	push True or Nil (for deeply or not)
	C7 00 11	Create foreach iterator
	A?		Assign it to a local variable.
	...		Perform other initialisation (depends on loop type)
	5F ?? ??	Goto end of loop check (Y:)
	X: ...		Code executed each time round the iteration.
	7?		Push the local variable referencing the iterator
	05		And move to the next thing to be iterated
	Y: 7?		Push the iterator again
	06		And check if we've finished iterating
	6F ?? ??	Go round the loop again if not (X:)


	"""
	pass


    def _BC_INCREMENTLOCAL(self, arg):
	"""

	This bytecode adds the given integer increment to the local
	variable in the Nth slot of the CodeBlock's argFrame and the
	returns both the increment and the new value of the local
	variable.  This instruction is used internally as part of the
	coding of for loops.

	"""

	pass


    def _BC_FORLOOPGOTO(self, arg):
	"""

	This instruction is placed at the end of a for loop construct.
	It takes the increment, current value of the counter (after
	having been incremented by 26N) and the limit value of the
	loop.  If the counter value now exceeds the loop limit,
	execution continues with the next instruction in sequence.  If
	it does not, execution passes to location N where N is the
	offset from the start of the block of bytecode.  This
	instruction is used internally as part of the coding of for
	loops.

	"""

	pass


    def _BC_FOREACHNEXT(self, arg):
	"""Do next iteration in foreach.

	Steps to next item in iteration.  See 307 000 021 - foreach
	for a full explanation."""

	pass


    def _BC_FOREACHCOMPLETE(self, arg):
	"""Checks if iteration complete.

	See 307 000 021 - foreach for a full explanation."""

	pass


#############################################################################

def main():
    """Test"""

    mem = "\x20" \
	  "\x22" \
	  "\x23" \
	  "\x24" \
	  "\x27\x00\x1a" \
	  "\x27\x02\x86" \
	  "\x27\xff\xf8" \
	  "\000" \
	  "\001" \
	  "\007\000\007"


    i = NewtonVM(mem)
    i.go()


#############################################################################

if __name__ == "__main__":
    main()


#############################################################################
